/*
Copyright (c) 2013, Silas Parker
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    Redistributions of source code must retain the above copyright notice, this
    list of conditions and the following disclaimer.
    Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation
    and/or other materials provided with the distribution.
    The name of Silas Parker may not be used to endorse or promote products
    derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/


#define WINVER 0x0500
#define _WIN32_WINNT 0x0500
#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <stdarg.h>
#include <sstream>

#include <scssdk_telemetry.h>
#include <eurotrucks2/scssdk_telemetry_eut2.h>

#include "serial.hpp"

static const float METERS_PER_SEC_TO_MILES_PER_HOUR = 2.2369f;
static const float METERS_PER_SEC_TO_KM_PER_HOUR = 3.6f;

// Game log file
scs_log_t game_log = NULL;

// COM port file
Serial serial_port;

unsigned long last_update = 0;


struct telemetry_state_t
{
	float	speed;
	float	rpm;
  bool left_ind;
  bool right_ind;
  bool parking_break;
  bool fuel_warning;
  int gear;
  float fuel;
  float fuel_capacity;
  
} telemetry;



SCSAPI_VOID telemetry_frame_end(const scs_event_t event, const void *const event_info, const scs_context_t context)
{
  if (!serial_port.is_valid())
    return;
    
  const unsigned long now = GetTickCount();
  const unsigned long diff = now - last_update;
  if (diff < 50)
    return;
    
  last_update = now;
  
  const float speed_mph = telemetry.speed * METERS_PER_SEC_TO_MILES_PER_HOUR;
  const float speed_kph = telemetry.speed * METERS_PER_SEC_TO_KM_PER_HOUR;
  
  
  Serial::formatted_data data;
  
  
  data.speed = (scs_u8_t)(abs(int(speed_mph * 2.0f)) & 0xFF);
  data.rpm = (scs_u8_t)(abs(int(telemetry.rpm / 15.0f)) & 0xFF);
  data.left_ind = telemetry.left_ind;
  data.right_ind = telemetry.right_ind;
  data.parking_break = telemetry.parking_break;
  data.fuel_warning = telemetry.fuel_warning;
  
  
  
  std::stringstream ss;
  
  ss.width(3);
  ss << abs(int(speed_mph));
  ss << " MPH";
  
  ss << "   G  ";
  
  if (telemetry.gear > 0)
  {
    ss << "D";
    ss.width(2);
    ss << telemetry.gear;
  }
  else if (telemetry.gear == 0)
  {
    ss << "N  ";
  }
  else
  {
    ss << "R";
    ss.width(2);
    ss << abs(telemetry.gear);
  }
  
  ss << "\n";
  
  ss.width(3);
  ss << abs(int(speed_kph));
  ss << " KPH";
  
  ss << "   F ";
  
  ss.width(3);
  ss << int(telemetry.fuel / telemetry.fuel_capacity * 100.0f) << "%";
  
  
  ss.width(3);
  ss << abs(int(speed_kph));
  
  data.text = ss.str();
  serial_port.write(data);
}



SCSAPI_VOID telemetry_store_float(const scs_string_t name, const scs_u32_t index, const scs_value_t *const value, const scs_context_t context)
{
	assert(value);
	assert(value->type == SCS_VALUE_TYPE_float);
	assert(context);
	*static_cast<float *>(context) = value->value_float.value;
}

SCSAPI_VOID telemetry_store_bool(const scs_string_t name, const scs_u32_t index, const scs_value_t *const value, const scs_context_t context)
{
	assert(value);
	assert(value->type == SCS_VALUE_TYPE_bool);
	assert(context);
	*static_cast<bool *>(context) = bool(value->value_bool.value);
}

SCSAPI_VOID telemetry_store_s32(const scs_string_t name, const scs_u32_t index, const scs_value_t *const value, const scs_context_t context)
{
	assert(value);
	assert(value->type == SCS_VALUE_TYPE_s32);
	assert(context);
	*static_cast<int *>(context) = value->value_s32.value;
}

SCSAPI_VOID telemetry_configuration(const scs_event_t event, const void *const event_info, const scs_context_t context)
{
  const struct scs_telemetry_configuration_t *const info = static_cast<const scs_telemetry_configuration_t *>(event_info);
  
  for (const scs_named_value_t *current = info->attributes; current->name; ++current)
  {
    if (strcmp(SCS_TELEMETRY_CONFIG_ATTRIBUTE_fuel_capacity, current->name) == 0)
    {
      telemetry.fuel_capacity = current->value.value_float.value;
    }
  }
}


SCSAPI_RESULT scs_telemetry_init(const scs_u32_t version, const scs_telemetry_init_params_t *const params)
{
	if (version != SCS_TELEMETRY_VERSION_1_00) {
		return SCS_RESULT_unsupported;
	}

	const scs_telemetry_init_params_v100_t *const version_params = static_cast<const scs_telemetry_init_params_v100_t *>(params);
  game_log = version_params->common.log;
  
  game_log(SCS_LOG_TYPE_message, "Plugin initialising");
  
  // char buffer[256];
  // GetCurrentDirectory(256, buffer);
  // game_log(SCS_LOG_TYPE_message, (std::string("Plugin CWD: ") + buffer).c_str());
  
  // Open COM port
  // TODO: Provide some way to specify port without recompiling
  std::string errmsg;
  if (!serial_port.open("COM3", errmsg))
  {
    game_log(SCS_LOG_TYPE_error, errmsg.c_str());
    return SCS_RESULT_generic_error;
  }
    
  serial_port.write_default();
  
  // Register for in game events
  bool registered =
    (version_params->register_for_event(
      SCS_TELEMETRY_EVENT_frame_end, telemetry_frame_end, NULL) == SCS_RESULT_ok) &&
    (version_params->register_for_event(
      SCS_TELEMETRY_EVENT_configuration, telemetry_configuration, NULL) == SCS_RESULT_ok) &&
    (version_params->register_for_channel(
      SCS_TELEMETRY_TRUCK_CHANNEL_speed, SCS_U32_NIL, SCS_VALUE_TYPE_float,
      SCS_TELEMETRY_CHANNEL_FLAG_none, telemetry_store_float, &telemetry.speed) == SCS_RESULT_ok) &&
    (version_params->register_for_channel(
      SCS_TELEMETRY_TRUCK_CHANNEL_engine_rpm, SCS_U32_NIL, SCS_VALUE_TYPE_float,
      SCS_TELEMETRY_CHANNEL_FLAG_none, telemetry_store_float, &telemetry.rpm) == SCS_RESULT_ok) &&
    (version_params->register_for_channel(
      SCS_TELEMETRY_TRUCK_CHANNEL_light_lblinker, SCS_U32_NIL, SCS_VALUE_TYPE_bool,
      SCS_TELEMETRY_CHANNEL_FLAG_none, telemetry_store_bool, &telemetry.left_ind) == SCS_RESULT_ok) &&
    (version_params->register_for_channel(
      SCS_TELEMETRY_TRUCK_CHANNEL_light_rblinker, SCS_U32_NIL, SCS_VALUE_TYPE_bool,
      SCS_TELEMETRY_CHANNEL_FLAG_none, telemetry_store_bool, &telemetry.right_ind) == SCS_RESULT_ok) &&
    (version_params->register_for_channel(
      SCS_TELEMETRY_TRUCK_CHANNEL_parking_brake, SCS_U32_NIL, SCS_VALUE_TYPE_bool,
      SCS_TELEMETRY_CHANNEL_FLAG_none, telemetry_store_bool, &telemetry.parking_break) == SCS_RESULT_ok) &&
    (version_params->register_for_channel(
      SCS_TELEMETRY_TRUCK_CHANNEL_fuel_warning, SCS_U32_NIL, SCS_VALUE_TYPE_bool,
      SCS_TELEMETRY_CHANNEL_FLAG_none, telemetry_store_bool, &telemetry.fuel_warning) == SCS_RESULT_ok) &&
    (version_params->register_for_channel(
      SCS_TELEMETRY_TRUCK_CHANNEL_engine_gear, SCS_U32_NIL, SCS_VALUE_TYPE_s32,
      SCS_TELEMETRY_CHANNEL_FLAG_none, telemetry_store_s32, &telemetry.gear) == SCS_RESULT_ok) &&
    (version_params->register_for_channel(
      SCS_TELEMETRY_TRUCK_CHANNEL_fuel, SCS_U32_NIL, SCS_VALUE_TYPE_float,
      SCS_TELEMETRY_CHANNEL_FLAG_none, telemetry_store_float, &telemetry.fuel) == SCS_RESULT_ok)
    ;
  
  if (!registered)
  {
    game_log(SCS_LOG_TYPE_error, "Unable to register callbacks");
		return SCS_RESULT_generic_error;
  }
  
  memset(&telemetry, 0, sizeof(telemetry));
  
  return SCS_RESULT_ok;
}

SCSAPI_VOID scs_telemetry_shutdown(void)
{
  serial_port.write_default();
  serial_port.close();

  game_log(SCS_LOG_TYPE_message, "Plugin shutdown");
	game_log = NULL;
}

BOOL APIENTRY DllMain(HMODULE module, DWORD reason_for_call, LPVOID reseved)
{
  if (reason_for_call == DLL_PROCESS_DETACH)
  {
		serial_port.close();
	}
	return TRUE;
}

