/*
Copyright (c) 2013, Silas Parker
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    Redistributions of source code must retain the above copyright notice, this
    list of conditions and the following disclaimer.
    Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation
    and/or other materials provided with the distribution.
    The name of Silas Parker may not be used to endorse or promote products
    derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include "serial.hpp"

#include <vector>


static const Serial::formatted_data default_data = {0, 0, false, false, false, false, std::string(" ") };


Serial::Serial() : com_port(INVALID_HANDLE_VALUE), prev_data(default_data)
{}

Serial::~Serial()
{
  close();
}

bool Serial::is_valid() const
{
  return com_port != INVALID_HANDLE_VALUE;
}

bool Serial::open(const std::string& name, std::string& errmsg)
{
  bool retval = false;
  DCB config;
  
  errmsg.clear();
  
  // If port already open, ignore call
  if (com_port != INVALID_HANDLE_VALUE)
  {
    errmsg = "COM port already open";
    goto exit; 
  }
  
  // Open COM port
  com_port = CreateFile(name.c_str(), GENERIC_READ | GENERIC_WRITE, 0, NULL,
    OPEN_EXISTING, 0, NULL);
  if (com_port == INVALID_HANDLE_VALUE)
  {
    errmsg = "Can not open COM port";
    goto exit;
  }
  
  // Get current config
  config.DCBlength = sizeof(config);
  if (!GetCommState(com_port, &config))
  {
    errmsg = "Can not get COM port config";
    goto exit;
  }
  
  // Set defaults
  config.BaudRate = CBR_115200;
  config.StopBits = ONESTOPBIT;
  config.Parity =   NOPARITY;
  config.ByteSize = 8;
  
  // Set config
  if (!SetCommState(com_port, &config))
  {
    errmsg = "Can not set COM port config";
    goto exit;
  }
  
  retval = true;

exit:
  if (retval == false)
  {
    close();
  }
  return retval;
}

void Serial::close()
{
  if (com_port != INVALID_HANDLE_VALUE)  
  {
    CloseHandle(com_port);
    com_port = INVALID_HANDLE_VALUE;
  }
}

void Serial::write(const formatted_data& data)
{
  if (data == prev_data)
    return;
    
  const bool same_text = (data.text == prev_data.text);
  
  if (com_port != INVALID_HANDLE_VALUE)  
  {
    const unsigned long buffer_size = 5;
    scs_u8_t buffer[buffer_size];
    buffer[0] = 0xFF;
    buffer[1] = data.speed;
    buffer[2] = data.rpm;
    buffer[3] = (scs_u8_t(data.fuel_warning) << 3) | (scs_u8_t(data.parking_break) << 2) | (scs_u8_t(data.left_ind) << 1) | scs_u8_t(data.right_ind);
    buffer[4] = same_text ? 0 : scs_u8_t(data.text.size());
    
    unsigned long wrote;
    WriteFile(com_port, buffer, buffer_size, &wrote, NULL);
    if (!same_text)
      WriteFile(com_port, data.text.data(), data.text.size(), &wrote, NULL);
    FlushFileBuffers(com_port);
    
    prev_data = data;
  }
}

void Serial::write_default()
{
  write(default_data);
}



